/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.matrix.solver;

import opale.tools.Debug;
import opale.mathtools.DVect;
import opale.matrix.*;

/**
* This class implements Cholesky decomposition according to the book "Analyse matricielle applique  l'art de l'ingnieur", Lascaux & Thodor, Ed. Masson.
* @author O.C.
* @date 04/2001
*/


public class SolverCholesky implements SymSolver, SymDecomposition
{
private double[] BtB;
private int n;
private boolean sdp = true;

public void decomp(SymMatrix A)
	{
	int cpti,cptj;
	double s;
	sdp = true;
	//double[] a = A.getArrayCopy();
	n = A.getDim();
	if (Decomposition.REPLACE) BtB=A.getInternalArray();
	else BtB = A.getArrayCopy();
	
	if (Debug.On) Debug.print("n = "+ n);
	
	
	for (int i=0;i<n;i++)
		{
		for (int j=i;j<n;j++)
			{
			cpti = i*(i+1)/2;
			cptj = j*(j+1)/2;
		
			//s = a[cpti+j];
			s = BtB[cptj+i];
			for (int k=0;k<i;k++)	s-=BtB[cpti++]*BtB[cptj++];
			
			if (i==j)	{sdp&=(s>0); BtB[cpti]=Math.sqrt(s);}
			else		BtB[cptj]=s/BtB[cpti];
			}
		}
	if (Debug.On) Debug.print("BtB = "+ BtB[0] +" " + BtB[1] +" "+ BtB[2] +" "+ BtB[3] +" "+ BtB[4] +" "+ BtB[5] +" ");
	
	}

/**
* Solve B tBx=b.
* @param Matrix A, not used.
*/
public void solve(SymMatrix A,DVect b, DVect x)
	{
      if (!sdp)  throw new IllegalArgumentException("Matrix must be symmetric positive definite !!");
     
	int i,k,cpti,dum;
	double s;
	double[] B = b.toArray();
	double[] X = x.toArray();

	for (i=0;i<n;i++)
		{
		cpti = i*(i+1)/2;
		s = B[i];
		for (k=0;k<i;k++) s-=BtB[cpti++]*X[k];
		X[i]=s/BtB[cpti];
		}

	for (i=n-1;i>=0;i--)
		{
		cpti = dum = (i+1)*i/2+i;
		s = X[i];
		for (k=i+1;k<n;k++) { cpti+=k; s-=BtB[cpti]*X[k]; }
		//X[i]=s/BtB[i*(i+1)/2+i];
		X[i]=s/BtB[dum];
		}
	}
/** 
* Compute the determinant : square  of product  B[i][i]
* @return     double, the determinant
* @since Opale matrix 0.11
*/
public double determinant() 
	{
	double det=1;
	for (int i = 0; i < n; i++) 
		det *= BtB[i*(i+1)/2+i];
	det *= det;
	return det;
	}


public static void main(String[] arg)
	{
/*	SymMatrix A = new SymMatrix(2);
	DVect b=new DVect(2);
	DVect x=new DVect(2);

	A.set(0,0,4);
	A.set(0,1,1);
	A.set(1,1,4);
	b.set(0,1);
	b.set(1,0);
	System.out.println(A);
	
	SolverCholesky ch = new SolverCholesky();
	ch.decomp(A);
	ch.solve(null,b,x);
	System.out.println(A);
	System.out.println(x);*/
	
	SymMatrix A = new SymMatrix(3);
	DVect b=new DVect(3);
	DVect x=new DVect(3);

	A.set(0,0,4);

	A.set(1,0,0);
	A.set(1,1,1);

	A.set(2,0,1);
	A.set(2,1,1);
	A.set(2,2,3);

	b.set(0,1);
	b.set(1,0);
	b.set(2,3);
	System.out.println(A);
	
	SolverCholesky ch = new SolverCholesky();
	ch.decomp(A);
	ch.solve(null,b,x);
	System.out.println("det = " +ch.determinant());
	System.out.println(A);
	System.out.println(x);
	}
	

}
