/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.mathtools;


/**
 * This class provides methods to manipulate the Bernstein polynomial.
 * A nth-degree Berstein polynomial is given by
 * B(i,n,u) = n! * u^i * (1-u)^(n-i) / i! * (n-i)!
 * with  0 <= i <= n
 * Initial version from R.P.
 * @author OpaleTeam
 * @date 05/2002
 */
public class Bernstein {

    
    /** 
     * Returns the evaluation,for a given value, of a Bernstein
     * polynom defined by the given arguments.
     * @param int i,
     * @param int n, the degree of the polynomial
     * @param double u, the value for which the polynomial will be computed
     * @return double, the computed value
     */
    public static double getValue(int i, int n, double u) {
    
	double[] temp = new double[n+1];
	
	/*for(int j=0; j<=n; j++) {
	  temp[j] = 0.0;
	  }*/
	
	temp[n-i] = 1.0;
	double u1 = 1.0 - u;
	
	for (int k = 1; k <= n; k++) {
	    for (int j = n; j >= k; j--) {
		temp[j] = u1 * temp[j] + u * temp[j-1];
	    }
	}
	return temp[n]; 
    }

      /** 
     * Returns all Bernstein polynomials evaluations,for a given
     * value, of a Bernstein polynom defined by the given arguments.
     * @param int n, the degree of the polynomial
     * @param double u, the value for which the polynomial will be computed
     * @return double, the computed value
     */
    public static double[] getAllValues(int n, double u) {
	double[] B = new double[n+1];
	B[0] = 1.0;
	double u1 = 1.0 - u;
	double saved;
	double temp;

	for (int i = 1; i <= n; i++) {
	    saved = 0.0;
	    for (int k = 0; k < i; k++) {
		temp = B[k];
		B[k] = saved + u1 * temp;
		saved = u * temp;
	    }
	    B[i] = saved;
	}
	
	return B;
    }


    


}
