/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.m2d.object2d;
import opale.tools.*;
import opale.mathtools.*;
import opale.m2d.*;
import java.util.Vector;
import java.io.*;


/**
* Cette classe modlise un nuage de points. Elle tend <code>Object2D</code> et implmente <code>Transformable</code>.
* Elle reprsente un nuage de points dans le plan et possede des mthodes de calculs sur ces points (drive, interpolation...)
* @author D.K. ; O.C.
* @since Opale-2d 0.1
*/
public class Data2D extends Object2D implements Transformable, Cloneable
{
protected double[] x,y;
protected int nbPts;

/**
* Constructeur par dfaut.
*/
public Data2D()
	{
	super();
	if (Debug.On) Debug.print("Dans Data2D()" + nbPts);
	x = new double[1];
	y = new double[1];
	
	}

/**
* Initialise l'objet avec 0 points et un repre OIJ donn.
* @param OIJ rep, le repre.
*/
public Data2D(OIJ rep)
        {
        super(rep);
	x = new double[1];
	y = new double[1];
        }

/**
* Initialise l'objet avec n points et un repre OIJ donn.
* @param OIJ rep, le repre.
* @param int n, le nombre de points
*/
public Data2D(OIJ rep, int n)
        {
        super(rep);
	nbPts = Math.abs(n);
	x = new double[nbPts];
	y = new double[nbPts];
        }

/**
* Le constructeur de copie.
* @param Data2D le nuage  copier.
*/
public Data2D(Data2D d)
        {
        super(d.getOIJ());
	int i;
	
        this.nbPts = d.nbPts;
	x = new double[nbPts];
	y = new double[nbPts];
	
	for (i=0;i<nbPts;i++)
		{
		x[i] = d.x[i];
		y[i] = d.y[i];
		}
	}
	
/**	
* Remplit l'objet avec le tableau de points pass en paramtre.
* @param int n, la taille des tableaux.
* @apram double[] x, tableau des x.
* @apram double[] y, tableau des y.
*/
public void set(int n, double[] x, double[] y)
	{
	int i;
	
        this.nbPts = n;
	this.x = new double[nbPts];
	this.y = new double[nbPts];
	for (i=0;i<nbPts;i++)
		{
		this.x[i] = x[i];
		this.y[i] = y[i];
		}
	}
	
/**
* Ajoute un point dans le tableau.
* double x,y le point  ajouter.
* @since Opale-2d 0.11
*/	
public void add(double x, double y)
	{
	int i;
	
	double[] xtemp = new double[nbPts];
	double[] ytemp = new double[nbPts];
	
	for (i=0;i<nbPts;i++)
		{
		xtemp[i] = this.x[i];
		ytemp[i] = this.y[i];
		}
		
	this.x = new double[++nbPts];		
	this.y = new double[nbPts];		

	for (i=0;i<nbPts-1;i++)
		{
		this.x[i] = xtemp[i];
		this.y[i] = ytemp[i];
		}
	this.x[nbPts] = x;
	this.y[nbPts] = y;
		
	}	
	
	
public void writeMore(PrintWriter f, OpaleSet p) throws InvalidFormatException
	{
	int i;
	f.println("N "+nbPts);
	f.println("Points ");
	for (i=0;i<nbPts;i++)
		f.println(x[i]+" "+y[i]);
	}	
	

	
public  int readKeyWord(String word, StreamReader f, OpaleSet p) throws java.io.IOException, InvalidFormatException
	{
	int type,i;
	if ( word.equals("N"))
		{
		nbPts = (int) f.nextDouble();
		x = new double[nbPts];
		y = new double[nbPts];
		return 0;
		}
	else if ( word.equals("Points"))
		{
		for (i=0;i<nbPts;i++)
			{
			x[i] = f.nextDouble();
			y[i] = f.nextDouble();
			}
		return 0;
		}
	return -1;
		
	}

/**
* Fixe la fonction a tracer, les bornes de dfinition et le nombre de points.
* @param IyFx f, la fonction  dessiner.
* @param double inf, sup, les bornes du domaine de dfinition.
* @param int n, le nombre de points  considerer sur la courbe.
* @since Opale-2d 0.11
*/
public void set(IyFx f, double inf, double sup, int n)
	{
	nbPts = Math.max(2,n); 
	
	if ( inf > sup )
		{
		double temp = sup;
		sup = inf;
		inf = temp;
		}
		
	x = new double[nbPts];
	y = new double[nbPts];
	int i;
	double pas = Math.abs(sup-inf)/(nbPts-1);
	
	for (i=0;i<nbPts;i++)
		{
		x[i] = inf + i*pas;
		y[i] = f.val(x[i]);
		}
	}

/**
* Fixe la fonction paramtrique  tracer, les bornes de dfinition et le nombre de points.
* @param IxyFt f, la fonction  dessiner.
* @param double tinf, tsup, les bornes du domaine de dfinition.
* @param int n, le nombre de points  considerer sur la courbe.
* @since Opale-2d 0.11
*/
public void set(IxyFt f, double tinf, double tsup, int n)
	{
	double t;
	nbPts = Math.max(2,n); 
	
	if ( tinf > tsup )
		{
		t = tsup;
		tsup = tinf;
		tinf = t;
		}
		
	x = new double[nbPts];
	y = new double[nbPts];
	int i;
	double pas = Math.abs(tsup-tinf)/(nbPts-1);
	
	for (i=0;i<nbPts;i++)
		{
		t =tinf + i*pas; 
		x[i] = f.x(t);;
		y[i] = f.y(t);
		}
	
	}


/**
* Cette mthode lit directement un fichier de donnes numriques et remplit l'objet courant en consquence.
* @param RFile f, le fichier de donnes.
*/	
public void readData(StreamReader f)  throws java.io.IOException, InvalidFormatException
	{
	int type,i;
	Vector xtemp = new Vector(100,20);
	Vector ytemp = new Vector(100,20);
	Double[] xtab, ytab;
	
	while (f.hasMoreTokens())
		{
		xtemp.add(new Double(f.nextDouble()));
		ytemp.add(new Double(f.nextDouble()));
		}
	nbPts = xtemp.size();
	x = new double[nbPts];
	y = new double[nbPts];
	
	xtab = new Double[nbPts];
	ytab = new Double[nbPts];
	
	xtemp.copyInto(xtab);
	ytemp.copyInto(ytab);
	
	for (i=0;i<nbPts;i++)
		{
		x[i] = xtab[i].doubleValue();	
		y[i] = ytab[i].doubleValue();	
		}
	}
	
// les mthodes de l'interface Transformable.

public void translate(double dx,double dy)
	{
	int i;
	for (i=0;i<nbPts;i++)
		{
		this.x[i]+=dx;
		this.y[i]+=dy;
		}
	}
	

public void rotate(double theta)
	{
	int i;
	final double costh=Math.cos(theta);
	final double sinth=Math.sin(theta);
	final OIJ rep = getOIJ();
	final double invsin = 1./Math.sin(rep.getTheta());
	final double tmp = Math.cos(rep.getTheta())*invsin*sinth;
	final double nI = Math.sqrt(rep.normSqI());
	final double nJ = Math.sqrt(rep.normSqJ());
	final double a00 =  costh - tmp;
	final double a01 =  -sinth*invsin/nI;
	final double a10 =   sinth*invsin/nJ;
	final double a11 =  costh + tmp;

	for (i=0;i<nbPts;i++)
		{
		double x=this.x[i];
		this.x[i]=x*a00+y[i]*a01 ;
		this.y[i]=x*a10+this.y[i]*a11 ;
		}
	}

public void rotate(double oriX, double oriY,double theta)
	{
	this.translate(-oriX,-oriY);
	this.rotate(theta);
	this.translate(oriX,oriY);
	}

public void transform(AffineTransform2D t)
	{
	t.transform(x,y,x,y,nbPts);
	}



/**
* Effectue un changement de repere de l'objet.
* @param OIJ rep, le nouveau repere.
*/
public void changeOIJ(OIJ rep)
	{
	int i;
	Matrix2D mv = new Matrix2D();
	getOIJ().matPassage(rep,mv);
	double[] p;
	for (i=0;i<nbPts;i++)
		{
		 p = mv.compute(x[i],y[i]);
		x[i] = p[0];
		y[i] = p[1];
		}
	setOIJ(rep);
	}
	
/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
*/
public Object clone()
	{
	int i;
	Data2D d = (Data2D) super.clone();
	d.x = new double[nbPts];
	d.y = new double[nbPts];
	
	for (i=0;i<nbPts;i++)
		{
		d.x[i] = x[i];
		d.y[i] = y[i];
		}
	return d;
   	}

/**
* Cette mthode teste si le point est gal  un objet pass en argument.
* @param Object obj un objet  comparer avec le point.
* @return <code>true</code> si l'objet  comparer est une instance de Point2D et
est gal au point courant; <code>false</code> sinon.
*/
public boolean equals(Object obj)
	{
	if (obj instanceof Data2D)
		{
        	Data2D d = (Data2D) obj;
		int i;
		boolean test=false;
		if ( (test=(nbPts == d.nbPts) ) )
			{
			for (i=0;i<nbPts;i++)
				{
				test = test && (x[i] == d.x[i]) && (y[i] == d.y[i]);
				}
			}
			
		return test && (getOIJ().equals(d.getOIJ()));
        	}
    	return false;
    }


public String toString()
	{
	String s="Data2D";
	return s;
	}
	

/**
* Drive le nuage de points en utilisant des diffrences 'avance'.
* @return Data2D, le rsultat obtenu.
*/	
public Data2D forwardDiff()
	{
	int i;
	double h;
	Data2D a = new Data2D(getOIJ(),nbPts-1);
	for(i=0;i<nbPts-1;i++)
		{
		a.x[i] = x[i];
		if ( (h=x[i+1]-x[i]) != 0)
			a.y[i] = (y[i+1]-y[i])/h;
		else {}//exception
			
		}
	return a;
	}
		
	

/**
* Drive le nuage de points en utilisant des diffrences 'retard'.
* @return Data2D, le rsultat obtenu.
*/	
public Data2D backwardDiff()
	{
	int i;
	double h;
	Data2D a = new Data2D(getOIJ(),nbPts-1);
	for(i=1;i<nbPts;i++)
		{
		a.x[i] = x[i];
		if ( (h=x[i]-x[i-1]) != 0)
			a.y[i] = (y[i]-y[i-1])/h;
		else {}//exception
			
		}
	return a;
	}

/**
* Drive le nuage de points en utilisant des diffrences 'centres'.
* @return Data2D, le rsultat obtenu.
*/	
public Data2D centeredDiff()
	{
	int i;
	double h;
	Data2D a = new Data2D(getOIJ(),nbPts-1);
	for(i=1;i<nbPts-1;i++)
		{
		a.x[i] = x[i];
		if ( (h=x[i+1]-x[i-1]) != 0)
			a.y[i] = (y[i+1]-y[i-1])/h;
		else {}//exception
			
		}
	return a;
	}

/**
* Calcule l'aire sou sle nuage de points par la mthode des rectangles 'avants'.
* @return double, le rsultat obtenu.
*/	
public double sumRectForward()
	{
	double sum=0.;
	int i;

	for(i=0;i<nbPts-1;i++)	
		sum+=y[i]*(x[i+1]-x[i]);
	
	return sum;	
	}
	
/**
* Calcule l'aire sou sle nuage de points par la mthode des rectangles 'retards'.
* @return double, le rsultat obtenu.
*/	
public double sumRectBackward()
	{
	double sum=0.;
	int i;

	for(i=1;i<nbPts;i++)	
		sum+=y[i]*(x[i]-x[i-1]);
	
	return sum;	
	}

/**
* Calcule l'aire sous le nuage de points par la mthode des rectangles 'centrs'.
* Cette mthode est quivalente  la mthode des trapzes.
* @return double, le rsultat obtenu.
*/	
public double sumRectCentered()
	{
	double sum=y[0]*(x[1]-x[0])/2;
	int i;

	for(i=1;i<nbPts-1;i++)	
		sum+=y[i]*(x[i+1]-x[i-1])/2;
	
	sum+=y[nbPts-1]*(x[nbPts-1]-x[nbPts-2])/2;
	return sum;	
	}

/**
* Calcule l'aire sous le nuage de points par la mthode des trapzes.
* @return double, le rsultat obtenu.
*/	
public double sumTrapez()
	{
	double sum=0.;
	int i;

	for(i=0;i<nbPts-1;i++)	
		sum+=(y[i]+y[i+1])/2*(x[i+1]-x[i]);
	
	return sum;	
	}

/**
* Cette mthode ajuste le nuage par une droite  y=mx+b.
* @return  double[], (m,b).
**/

public double[] regLin()
        {

  double[] p=new double[2];
  double m,b;
  double sommX = 0;
  double sommY = 0;
  double sommXY = 0;
  double sommXX = 0;

  for(int i=0 ;i<this.nbPts ;i++)
    {
     sommX+=this.x[i];
     sommY+=this.y[i];
     sommXY+=this.x[i]*this.y[i];
     sommXX+=this.x[i]*this.x[i];
    }

   m=(nbPts*sommXY-sommX*sommY)/(nbPts*sommXX-sommX*sommX);

   b=(sommY*sommXX-sommXY*sommX)/(nbPts*sommXX-sommX*sommX);
   
   p[0] = m;
   p[1] = b;
   return   p;

        }
/**
* Cette mthode renvoie une instance de Line2D correspondant  la regression linaire du nuage.
* @return  Line2D, la droite de regression.
**/

public Line2D regLinear()
        {

  double m,b;
  double sommX = 0;
  double sommY = 0;
  double sommXY = 0;
  double sommXX = 0;

  for(int i=0 ;i<this.nbPts ;i++)
    {
     sommX+=this.x[i];
     sommY+=this.y[i];
     sommXY+=this.x[i]*this.y[i];
     sommXX+=this.x[i]*this.x[i];
    }

   m=(nbPts*sommXY-sommX*sommY)/(nbPts*sommXX-sommX*sommX);

   b=(sommY*sommXX-sommXY*sommX)/(nbPts*sommXX-sommX*sommX);
   
   
   Line2D lin = null;
   try
   {
  lin = new Line2D(getOIJ(),m,-1,b);
   }
   catch(LineException e)
   {
   System.err.println("Erreur dans Line2D.regLin() !!");
   System.exit(-1); // ne devrait jamais arriver
   }
   return   lin;

        }
		
}
