/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.common.keymaster.rest.api.service;

import io.swagger.v3.oas.annotations.headers.Header;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.security.SecurityRequirements;
import io.swagger.v3.oas.annotations.tags.Tag;
import java.io.Serializable;
import java.util.List;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.apache.syncope.common.keymaster.client.api.model.Domain;
import org.apache.syncope.common.lib.types.CipherAlgorithm;
import org.apache.syncope.common.rest.api.RESTHeaders;

/**
 * REST operations for Self Keymaster's domains.
 */
@Tag(name = "Domains")
@SecurityRequirements({
    @SecurityRequirement(name = "BasicAuthentication") })
@Path("domains")
public interface DomainService extends Serializable {

    /**
     * Returns the list of defined domains.
     *
     * @return list of defined domains
     */
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    List<Domain> list();

    /**
     * Returns the domain matching the given key.
     *
     * @param key key of the domain to be read
     * @return domain matching the given key
     */
    @GET
    @Path("{key}")
    @Produces({ MediaType.APPLICATION_JSON })
    Domain read(@NotNull @PathParam("key") String key);

    /**
     * Creates a new domain.
     *
     * @param domain domain to be created
     * @return Response object featuring Location header of created domain
     */
    @ApiResponses(
            @ApiResponse(responseCode = "201",
                    description = "Domain successfully created", headers = {
                @Header(name = RESTHeaders.RESOURCE_KEY, schema =
                        @Schema(type = "string"),
                        description = "Key of the domin created"),
                @Header(name = HttpHeaders.LOCATION, schema =
                        @Schema(type = "string"),
                        description = "URL of the domin created") }))
    @POST
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    Response create(Domain domain);

    /**
     * Change admin's password for the given domain.
     *
     * @param key key of domain to be updated
     * @param password encoded password value
     * @param cipherAlgorithm password cipher algorithm
     */
    @ApiResponses(
            @ApiResponse(responseCode = "204", description = "Operation was successful"))
    @POST
    @Path("{key}/changeAdminPassword")
    @Produces({ MediaType.APPLICATION_JSON })
    void changeAdminPassword(
            @NotNull @PathParam("key") String key,
            @QueryParam("password") String password,
            @QueryParam("cipherAlgorithm") CipherAlgorithm cipherAlgorithm);

    /**
     * Adjusts the connection pool to the domain database.
     *
     * @param key key of domain to be updated
     * @param poolMaxActive database pool max size
     * @param poolMinIdle database pool max size
     */
    @ApiResponses(
            @ApiResponse(responseCode = "204", description = "Operation was successful"))
    @POST
    @Path("{key}/adjustPoolSize")
    @Produces({ MediaType.APPLICATION_JSON })
    void adjustPoolSize(
            @NotNull @PathParam("key") String key,
            @QueryParam("poolMaxActive") int poolMaxActive,
            @QueryParam("poolMinIdle") int poolMinIdle);

    /**
     * Deletes the domain matching the provided key.
     *
     * @param key key of domain to be deleted
     */
    @ApiResponses(
            @ApiResponse(responseCode = "204", description = "Operation was successful"))
    @DELETE
    @Path("{key}")
    @Produces({ MediaType.APPLICATION_JSON })
    void delete(@NotNull @PathParam("key") String key);
}
